from PySide6.QtWidgets import QWidget, QVBoxLayout, QLabel, QLineEdit, QPushButton, QMessageBox, QHBoxLayout, QFrame, \
    QCheckBox
from PySide6.QtCore import Qt, QPropertyAnimation, QEasingCurve, QTimer, QSettings
from PySide6.QtGui import QFont, QIcon, QPixmap, QPainter, QLinearGradient, QColor
import hashlib
import sqlite3
import os
import sys
from utils.path_utils import get_db_path

# Dapatkan path database
BASE_DIR = getattr(sys, '_MEIPASS', os.path.abspath(os.path.dirname(sys.argv[0])))
DB_PATH = get_db_path()


class GradientWidget(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)

    def paintEvent(self, event):
        painter = QPainter(self)
        gradient = QLinearGradient(0, 0, self.width(), self.height())
        gradient.setColorAt(0, QColor(74, 144, 226))  # Biru muda
        gradient.setColorAt(1, QColor(41, 98, 255))  # Biru tua
        painter.fillRect(self.rect(), gradient)


class LoginWindow(QWidget):
    def __init__(self, controller):
        super().__init__()
        self.controller = controller
        self.settings = QSettings("MayaGrahaKencana", "POS_System")
        self.setup_animations()
        self.init_ui()
        self.load_saved_credentials()

    def setup_animations(self):
        self.shake_animation = QPropertyAnimation(self, b"pos")
        self.shake_animation.setEasingCurve(QEasingCurve.OutCubic)
        self.shake_animation.setDuration(500)

    def load_saved_credentials(self):
        """Load saved username dan remember me setting"""
        remember_me = self.settings.value("remember_me", False, type=bool)
        if remember_me:
            username = self.settings.value("saved_username", "")
            self.input_username.setText(username)
            self.checkbox_remember.setChecked(True)
            self.input_password.setFocus()

    def init_ui(self):
        # Setup window properties
        self.setWindowTitle("POS System - Login")
        self.setFixedSize(450, 600)
        self.setStyleSheet("""
            QWidget {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                    stop:0 #4a90e2, stop:1 #2962ff);
                font-family: 'Segoe UI', Arial, sans-serif;
            }
        """)

        # Main layout dengan margin dan spacing
        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(40, 40, 40, 40)
        main_layout.setSpacing(0)
        main_layout.setAlignment(Qt.AlignCenter)

        # Container untuk konten login (card effect)
        self.login_container = QFrame()
        self.login_container.setStyleSheet("""
            QFrame {
                background: white;
                border-radius: 20px;
                padding: 0px;
            }
        """)
        self.login_container.setFixedSize(370, 520)  # Tinggi ditambah untuk accommodate checkbox

        # Layout untuk container
        container_layout = QVBoxLayout(self.login_container)
        container_layout.setContentsMargins(40, 40, 40, 40)
        container_layout.setSpacing(25)
        container_layout.setAlignment(Qt.AlignCenter)

        # Header section
        header_layout = QVBoxLayout()
        header_layout.setAlignment(Qt.AlignCenter)
        header_layout.setSpacing(2)

        # Icon/Logo
        self.logo_label = QLabel()
        self.logo_label.setFixedSize(80, 80)
        self.logo_label.setStyleSheet("""
            QLabel {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                    stop:0 #4a90e2, stop:1 #2962ff);
                border-radius: 40px;
                color: red;
                font-size: 24px;
                font-weight: bold;
                qproperty-alignment: AlignCenter;                
            }
        """)
        self.logo_label.setText("POS")

        # Title
        title_label = QLabel("Welcome Back")
        title_label.setStyleSheet("""
            QLabel {
                color: #2c3e50;
                font-size: 28px;
                font-weight: bold;
                qproperty-alignment: AlignCenter;
            }
        """)

        # Subtitle
        subtitle_label = QLabel("Sign in to your account")
        subtitle_label.setStyleSheet("""
            QLabel {
                color: #7f8c8d;
                font-size: 14px;
                qproperty-alignment: AlignCenter;
            }
        """)

        header_layout.addWidget(self.logo_label)
        header_layout.addSpacing(10)  # Jarak tambahan khusus sebelum title
        header_layout.addWidget(title_label)
        header_layout.addWidget(subtitle_label)

        # Form section
        form_layout = QVBoxLayout()
        form_layout.setSpacing(40)

        # Username field
        username_layout = QVBoxLayout()
        username_layout.setSpacing(8)

        username_label = QLabel("Username")
        username_label.setStyleSheet("""
            QLabel {
                color: #2c3e50;
                font-size: 13px;
                font-weight: 600;
            }
        """)

        self.input_username = QLineEdit()
        self.input_username.setPlaceholderText("Enter your username")
        self.input_username.setStyleSheet("""
            QLineEdit {
                background: #f8f9fa;
                border: 2px solid #e9ecef;
                border-radius: 12px;
                padding: 10px;
                font-size: 16px;
                color: #2c3e50;
            }
            QLineEdit:focus {
                border-color: #4a90e2;
                background: white;
            }
            QLineEdit:hover {
                border-color: #adb5bd;
            }
        """)
        self.input_username.setMinimumHeight(45)

        username_layout.addWidget(username_label)
        username_layout.addWidget(self.input_username)

        # Password field
        password_layout = QVBoxLayout()
        password_layout.setSpacing(8)

        password_label = QLabel("Password")
        password_label.setStyleSheet("""
            QLabel {
                color: #2c3e50;
                font-size: 13px;
                font-weight: 600;
            }
        """)

        # Container untuk password input + show/hide button
        password_input_container = QHBoxLayout()
        password_input_container.setSpacing(5)

        self.input_password = QLineEdit()
        self.input_password.setPlaceholderText("Enter your password")
        self.input_password.setEchoMode(QLineEdit.Password)
        self.input_password.setStyleSheet("""
            QLineEdit {
                background: #f8f9fa;
                border: 2px solid #e9ecef;
                border-radius: 12px;
                padding: 10px;
                font-size: 16px;
                color: #2c3e50;
            }
            QLineEdit:focus {
                border-color: #4a90e2;
                background: white;
            }
            QLineEdit:hover {
                border-color: #adb5bd;
            }
        """)
        self.input_password.setMinimumHeight(45)
        self.input_password.returnPressed.connect(self.check_login)

        # Show/Hide Password Button
        self.btn_toggle_password = QPushButton("👁")
        self.btn_toggle_password.setStyleSheet("""
            QPushButton {
                background: #f8f9fa;
                border: 2px solid #e9ecef;
                border-radius: 12px;
                padding: 10px;
                font-size: 14px;
                color: #7f8c8d;
                min-width: 50px;
                max-width: 50px;
            }
            QPushButton:hover {
                background: #e9ecef;
                border-color: #adb5bd;
            }
            QPushButton:pressed {
                background: #dee2e6;
            }
        """)
        self.btn_toggle_password.setCursor(Qt.PointingHandCursor)
        self.btn_toggle_password.setCheckable(True)
        self.btn_toggle_password.clicked.connect(self.toggle_password_visibility)

        password_input_container.addWidget(self.input_password)
        password_input_container.addWidget(self.btn_toggle_password)

        password_layout.addWidget(password_label)
        password_layout.addLayout(password_input_container)

        # Add fields to form
        form_layout.addLayout(username_layout)
        form_layout.addLayout(password_layout)

        # Remember Me Checkbox
        self.checkbox_remember = QCheckBox("Remember my username")
        self.checkbox_remember.setStyleSheet("""
            QCheckBox {
                color: #7f8c8d;
                font-size: 13px;
                spacing: 8px;
            }
            QCheckBox::indicator {
                width: 16px;
                height: 16px;
                border: 2px solid #bdc3c7;
                border-radius: 4px;
                background: white;
            }
            QCheckBox::indicator:checked {
                background: #4a90e2;
                border-color: #4a90e2;
            }
            QCheckBox::indicator:checked:hover {
                background: #3a80d2;
                border-color: #3a80d2;
            }
            QCheckBox::indicator:hover {
                border-color: #4a90e2;
            }
        """)
        self.checkbox_remember.setCursor(Qt.PointingHandCursor)

        # Login button
        self.button_login = QPushButton("Sign In")
        self.button_login.setStyleSheet("""
            QPushButton {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                    stop:0 #4a90e2, stop:1 #2962ff);
                color: white;
                border: none;
                border-radius: 12px;
                padding: 16px;
                font-size: 16px;
                font-weight: bold;
                margin-top: 10px;
            }
            QPushButton:hover {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                    stop:0 #3a80d2, stop:1 #1a52ef);
            }
            QPushButton:pressed {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                    stop:0 #2a70c2, stop:1 #0a42df);
            }
            QPushButton:disabled {
                background: #bdc3c7;
                color: #7f8c8d;
            }
        """)
        self.button_login.setCursor(Qt.PointingHandCursor)
        self.button_login.setMinimumHeight(50)
        self.button_login.clicked.connect(self.check_login)

        # Footer
        footer_label = QLabel("POS System v1.0 • © 2024")
        footer_label.setStyleSheet("""
            QLabel {
                color: #95a5a6;
                font-size: 11px;
                qproperty-alignment: AlignCenter;
                margin-top: 10px;
            }
        """)

        # Add all to container
        container_layout.addLayout(header_layout)
        container_layout.addLayout(form_layout)
        container_layout.addWidget(self.checkbox_remember)
        container_layout.addWidget(self.button_login)
        container_layout.addWidget(footer_label)

        # Add container to main layout
        main_layout.addWidget(self.login_container)
        self.setLayout(main_layout)

        # Set focus ke username field
        self.input_username.setFocus()

        # Loading state
        self.is_loading = False

    def toggle_password_visibility(self):
        """Toggle between showing and hiding password"""
        if self.btn_toggle_password.isChecked():
            self.input_password.setEchoMode(QLineEdit.Normal)
            self.btn_toggle_password.setText("🔒")
        else:
            self.input_password.setEchoMode(QLineEdit.Password)
            self.btn_toggle_password.setText("👁")

    def reset_toggle_password(self):
        """Reset toggle password button ke state awal"""
        self.btn_toggle_password.setChecked(False)
        self.input_password.setEchoMode(QLineEdit.Password)
        self.btn_toggle_password.setText("👁")

    def shake_window(self):
        """Animasi shake ketika login gagal"""
        original_pos = self.pos()
        self.shake_animation.setStartValue(original_pos)

        # Buat efek shake
        points = [
            original_pos + self.mapToParent(self.rect().topLeft()) * 0.02,
            original_pos - self.mapToParent(self.rect().topLeft()) * 0.02,
            original_pos + self.mapToParent(self.rect().topLeft()) * 0.01,
            original_pos
        ]

        self.shake_animation.setKeyValueAt(0.25, points[0])
        self.shake_animation.setKeyValueAt(0.5, points[1])
        self.shake_animation.setKeyValueAt(0.75, points[2])
        self.shake_animation.setEndValue(points[3])

        self.shake_animation.start()

    def set_loading_state(self, loading):
        """Set state loading untuk button"""
        self.is_loading = loading
        self.button_login.setEnabled(not loading)

        if loading:
            self.button_login.setText("Signing In...")
        else:
            self.button_login.setText("Sign In")

    def check_login(self):
        if self.is_loading:
            return

        username = self.input_username.text().strip()
        password = self.input_password.text().strip()

        # Validasi input
        if not username or not password:
            self.show_error("Please enter both username and password")
            self.shake_window()
            return

        # Set loading state
        self.set_loading_state(True)

        # Simulate processing delay
        QTimer.singleShot(800, lambda: self.process_login(username, password))

    def process_login(self, username, password):
        try:
            # Hash password dengan MD5
            password_hash = hashlib.md5(password.encode()).hexdigest()

            # Cek kredensial di database
            conn = sqlite3.connect(DB_PATH)
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()

            cursor.execute("""
                SELECT id, nama, password 
                FROM per_employee 
                WHERE nama = ? AND password = ?
            """, (username, password_hash))

            user = cursor.fetchone()
            conn.close()

            if user:
                user_info = {
                    "id": user["id"],
                    "nama": user["nama"]
                }
                print("🟢 Login berhasil:", user_info)

                # Save credentials jika remember me dicentang
                if self.checkbox_remember.isChecked():
                    self.settings.setValue("remember_me", True)
                    self.settings.setValue("saved_username", username)
                else:
                    self.settings.setValue("remember_me", False)
                    self.settings.setValue("saved_username", "")

                # Success animation
                self.show_success()

                # Panggil controller untuk proses login berhasil
                QTimer.singleShot(600, lambda: self.controller.login_success(user_info))

            else:
                self.set_loading_state(False)
                self.show_error("Invalid username or password")
                self.shake_window()

        except sqlite3.Error as e:
            self.set_loading_state(False)
            print("❌ Database error:", e)
            self.show_error(f"Database error: {str(e)}")
            self.shake_window()
        except Exception as e:
            self.set_loading_state(False)
            print("❌ Unexpected error:", e)
            self.show_error(f"Unexpected error: {str(e)}")
            self.shake_window()

    def show_error(self, message):
        """Tampilkan error message dengan style yang bagus"""
        msg = QMessageBox(self)
        msg.setIcon(QMessageBox.Warning)
        msg.setWindowTitle("Login Failed")
        msg.setText(message)
        msg.setStyleSheet("""
            QMessageBox {
                background: white;
                border-radius: 15px;
            }
            QMessageBox QPushButton {
                background: #e74c3c;
                color: white;
                border: none;
                border-radius: 8px;
                padding: 8px 16px;
                font-weight: bold;
            }
            QMessageBox QPushButton:hover {
                background: #c0392b;
            }
        """)
        msg.exec()

    def show_success(self):
        """Animasi success sebelum pindah window"""
        self.login_container.setStyleSheet("""
            QFrame {
                background: white;
                border-radius: 20px;
                border: 3px solid #2ecc71;
                padding: 0px;
            }
        """)
        self.button_login.setStyleSheet("""
            QPushButton {
                background: #2ecc71;
                color: white;
                border: none;
                border-radius: 12px;
                padding: 16px;
                font-size: 16px;
                font-weight: bold;
                margin-top: 10px;
            }
        """)
        self.button_login.setText("✓ Login Successful!")

    def clear_fields(self):
        """Bersihkan field input"""
        # Hanya clear password, username tetap jika remember me dicentang
        self.input_password.clear()
        self.reset_toggle_password()
        if not self.checkbox_remember.isChecked():
            self.input_username.clear()
        self.input_username.setFocus() if not self.checkbox_remember.isChecked() else self.input_password.setFocus()
        self.set_loading_state(False)

    def keyPressEvent(self, event):
        """Handle keyboard events"""
        if event.key() == Qt.Key_Escape:
            self.close()
        elif event.key() == Qt.Key_Return or event.key() == Qt.Key_Enter:
            self.check_login()
        else:
            super().keyPressEvent(event)